<?php

namespace App\Jobs;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Config;
use Carbon\Carbon;

class SendRejectedInvoicesSummaryJob extends Job
{
    public function handle()
    {
        $connections = $this->getConnections();
        $recipients = $this->getRecipients();

        if ($connections->isEmpty()) {
            echo "No hay conexiones habilitadas\n";
            return;
        }

        if ($recipients->isEmpty()) {
            echo "No hay destinatarios activos\n";
            return;
        }

        $groupedInvoices = [];  
        $toRegister = [];   

        foreach ($connections as $connection) {

            echo sprintf(
                "Empresa: %s | DB: %s | Host: %s\n",
                $connection->sub_dominio,
                $connection->db_name,
                $connection->db_host
            );

            $rejectedInvoices = $this->getRejectedInvoicesFromCompany($connection);

            if ($rejectedInvoices->isEmpty()) {
                continue;
            }

            // Filtrar solo los NO notificados
            $newRejected = $rejectedInvoices->filter(function ($invoice) use ($connection) {
                return !DB::connection('mysql')
                    ->table('notified_rejected_invoices')
                    ->where('company_id', $connection->id_empresa)
                    ->where('invoice_id', $invoice->invoice_id)
                    ->exists();
            });

            if ($newRejected->isEmpty()) {
                continue;
            }

            $groupedInvoices[$connection->sub_dominio] = [
                'company' => $connection,
                'invoices' => $newRejected
            ];

            foreach ($newRejected as $invoice) {
                $toRegister[] = [
                    'company_id' => $connection->id_empresa,
                    'company_db' => $connection->db_name,
                    'invoice_id' => $invoice->invoice_id,
                    'invoice_table_id' => $invoice->id ?? null,
                    'error_code' => $invoice->error_code,
                    'invoice_date' => $invoice->date,
                    'notified_at' => date('Y-m-d H:i:s'),
                ];
            }
        }

        if (empty($groupedInvoices)) {
            echo "No hay nuevos comprobantes para notificar\n";
            return;
        }

        $this->sendGroupedEmail($groupedInvoices, $recipients);

        DB::connection('mysql')
            ->table('notified_rejected_invoices')
            ->insert($toRegister);

        echo "Correo enviado con todas las empresas agrupadas\n";
    }

    private function getRecipients()
    {
        return DB::connection('mysql') // conexión default
            ->table('email_recipients')
            ->select('first_name', 'last_name', 'email')
            ->where('is_active', 1)
            ->get();
    }

    private function getConnections()
    {
        // IDs habilitados desde DB procesos-batch
        $enabledConnectionIds = DB::connection('mysql')
            ->table('batch_connections')
            ->where('is_enabled', 1)
            ->pluck('admin_connection_id');

        if ($enabledConnectionIds->isEmpty()) {
            return collect();
        }

        return DB::connection('admin')
            ->table('empresas')
            ->select(
                'id_empresa',
                'sub_dominio',
                'db_host',
                'db_name',
                'db_usuario',
                'db_contrasena'
            )
            ->whereIn('id_empresa', $enabledConnectionIds)
            ->where('estado', 1)
            ->where('status', 1)
            ->get();
    }

    private function getRejectedInvoicesFromCompany($connection)
    {
        $connectionName = 'company_' . $connection->id_empresa;

        Config::set("database.connections.$connectionName", [
            'driver' => 'mysql',
            'host' => $connection->db_host,
            'database' => $connection->db_name,
            'username' => $connection->db_usuario,
            'password' => $connection->db_contrasena,
            'charset' => 'utf8mb4',
            'collation' => 'utf8mb4_unicode_ci',
            'prefix' => '',
            'strict' => false,
        ]);


        $start = Carbon::today('America/Lima')->startOfDay()->toDateTimeString();
        $end = Carbon::today('America/Lima')->endOfDay()->toDateTimeString();

        echo "Rango de fechas: {$start} - {$end}\n";

        $sql = "
            SELECT
                id,
                invoice_id,
                flg_response,
                error_code,
                date
            FROM tec_sales
            WHERE flg_response = ?
            AND error_code <> ?
            AND date BETWEEN ? AND ?
        ";

        return collect(
            DB::connection($connectionName)->select(
                $sql,
                [
                    0,
                    '0',
                    $start,
                    $end,
                ]
            )

        );
    }

    private function sendGroupedEmail(array $groupedInvoices, $recipients)
    {
        $html = '
    <html>
    <head>
        <style>
            body {
                font-family: Arial, Helvetica, sans-serif;
                font-size: 14px;
                color: #333;
            }
            h2 {
                background: #6f42c1;
                color: #fff;
                padding: 10px;
            }
            h3 {
                margin-top: 25px;
                color: #6f42c1;
            }
            table {
                width: 100%;
                border-collapse: collapse;
                margin-top: 10px;
            }
            th, td {
                border: 1px solid #ddd;
                padding: 8px;
                text-align: left;
            }
            th {
                background-color: #f3f3f3;
            }
            tr:nth-child(even) {
                background-color: #fafafa;
            }
            .footer {
                margin-top: 30px;
                font-size: 12px;
                color: #777;
            }
        </style>
    </head>
    <body>

    <h2>Resumen de comprobantes rechazados</h2>
    <p>Este es un resumen automático de los comprobantes rechazados detectados.</p>
    ';

        foreach ($groupedInvoices as $group) {

            $company = $group['company'];
            $invoices = $group['invoices'];

            $html .= "<h3>Empresa: {$company->sub_dominio}</h3>";
            $html .= "
        <table>
            <thead>
                <tr>
                    <th>Invoice ID</th>
                    <th>Código de error</th>
                    <th>Fecha</th>
                </tr>
            </thead>
            <tbody>
        ";

            foreach ($invoices as $inv) {
                $html .= "
                <tr>
                    <td>{$inv->invoice_id}</td>
                    <td>{$inv->error_code}</td>
                    <td>{$inv->date}</td>
                </tr>
            ";
            }

            $html .= "
            </tbody>
        </table>
        ";
        }

        $html .= '
        <div class="footer">
            <p>Este correo fue generado automáticamente. No responder.</p>
        </div>

    </body>
    </html>
    ';

        Mail::html($html, function ($message) use ($recipients) {
            $message->to($recipients->pluck('email')->toArray())
                ->subject('Resumen de comprobantes rechazados');
        });
    }

}
