<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DTOs\PurchaseOrder\PurchaseOrderCreateDto;
use App\DTOs\PurchaseOrder\PurchaseOrderUpdateDto;
use App\Repositories\Interfaces\IPurchaseOrderRepository;
use Exception;
use App\Models\PurchaseOrderHeader;
use App\Models\PurchaseOrderItems;
use Symfony\Component\HttpKernel\Controller\ArgumentResolver\RequestValueResolver;

class PurchaseOrderController extends Controller
{

    private $purchaseOrderRepository;
    private $errorConnection = "";

    public function __construct(Request $request, IPurchaseOrderRepository $purchaseOrderRepository)
    {
        if (isset($request['business_information'])) {
            $business_information = $request['business_information'];
            $ruc = $business_information['ruc_number'];

            $existe_conexion = $this->setDBClient($ruc);

            if ($existe_conexion['error']) {
                $this->errorConnection = $existe_conexion['message'];
                return $this->messageError("Se espera un número de RUC válido.", 201);
            } else {
                $this->purchaseOrderRepository = $purchaseOrderRepository;
            }
        } else {
            $this->errorConnection = 'Error en la conexión. Se espera número de RUC';
        }
    }

    public function index(Request $request)
    {
        if ($this->errorConnection != '') {
            return $this->messageError($this->errorConnection, 201);
        }

        $result = $this->purchaseOrderRepository->paginate($request->all());

        if ($result) {
            return $this->successResponse($result, "Ordenes de compra encontradas correctamente", 201);
        } else {
            return $this->errorResponse("No hay Ordenes de compra", 201);
        }
    }

    public function show(int $id, Request $request)
    {
        $business_information = $request['business_information'];
        $business_ruc = $business_information['ruc_number'];
        if ($this->errorConnection != '') {
            return $this->messageError($this->errorConnection, 500);
        }

        $result = $this->purchaseOrderRepository->find($id, $business_ruc);

        if ($result) {
            return $this->successResponse($result, "Orden de compra encontrada correctamente", 200);
        } else {
            return $this->errorResponse("Orden de compra no encontrada", 200);
        }
    }

    public function store(Request $request)
    {
        try {
            if ($this->errorConnection != '') {
                return $this->messageError($this->errorConnection, 500);
            }

            $business_information = $request['business_information'];
            $business_ruc = $business_information['ruc_number'];

            $this->validate(
                $request,
                [
                    'date' => 'required|date',
                    'invoice_id' => 'nullable|string|max:12',
                    'supplier_id' => 'required|integer',
                    'supplier_name' => 'required|string|max:55',
                    'currency' => 'nullable|string|max:3',
                    'exchange' => 'nullable|numeric',
                    'affected' => 'required|numeric',
                    'non_affected' => 'required|numeric',
                    'exonerated' => 'required|numeric',
                    'total' => 'required|numeric',
                    'product_discount' => 'nullable|numeric',
                    'order_discount_id' => 'nullable|string|max:20',
                    'order_discount' => 'nullable|numeric',
                    'total_discount' => 'nullable|numeric',
                    'product_tax' => 'nullable|numeric',
                    'order_tax_id' => 'nullable|string|max:20',
                    'order_tax' => 'nullable|numeric',
                    'total_tax' => 'nullable|numeric',
                    'tax_ICBPER' => 'required|numeric',
                    'mult_ICBPER' => 'required|numeric',
                    'subtotal' => 'required|numeric',
                    'grand_total' => 'required|numeric',
                    'total_items' => 'nullable|integer',
                    'total_quantity' => 'nullable|numeric',
                    'note' => 'nullable|string|max:1000',
                    'rounding' => 'nullable|numeric',
                    'attachment' => 'nullable|string|max:255',
                    'expiration_date' => 'nullable|date',
                    'canal_id' => 'nullable|integer',
                    'payment_method_id' => 'nullable|integer',
                    'estado' => 'required|integer',
                    'status' => 'nullable|integer',

                    'items.*.purchase_order_header_id' => 'required|integer|max:15',
                    'items.*.product_id' => 'integer|nullable', //Este campo debe validarse que exista
                    'items.*.code' => 'string|nullable|max:50',
                    'items.*.name' => 'string|nullable|max:255',
                    'items.*.quantity' => 'required|numeric',
                    'items.*.unit_price' => 'required|numeric',
                    'items.*.affect_price' => 'required|numeric',
                    'items.*.non_affected_price' => 'required|numeric',
                    'items.*.exonerated_price' => 'required|numeric',
                    'items.*.gratuito_price' => 'numeric|nullable',
                    'items.*.price_affected_item' => 'required|numeric',
                    'items.*.price_no_affected_item' => 'required|numeric',
                    'items.*.price_exonerated_item' => 'required|numeric',
                    'items.*.price_gratuito_item' => 'numeric|nullable',
                    'items.*.discount' => 'string|nullable|max:20',
                    'items.*.item_discount' => 'numeric|nullable',
                    'items.*.prr_discount' => 'required|numeric',
                    'items.*.tax' => 'required|numeric',
                    'items.*.item_tax' => 'numeric|nullable',
                    'items.*.tax_gratuito' => 'numeric|nullable',
                    'items.*.item_tax_gratuito' => 'numeric|nullable',
                    'items.*.igv' => 'required|string|max:20',
                    'items.*.tax_ICBPER' => 'required|numeric',
                    'items.*.quantity_ICBPER' => 'required|integer',
                    'items.*.subtotal' => 'required|numeric',
                    'items.*.real_unit_price' => 'numeric|nullable',
                    'items.*.cost' => 'numeric|nullable',
                    'items.*.tax_method' => 'required|integer',
                    'items.*.status' => 'nullable|integer',
                ],
                [   //Mensajes para las validaciones:
                    'date.required' => 'El campo Fecha es obligatorio.',
                    'date.date' => 'El campo Fecha debe ser una fecha válida.',
                    'invoice_id.max' => 'El campo ID de factura no debe superar los 12 caracteres.',
                    'supplier_id.required' => 'El campo ID de proveedor es obligatorio.',
                    'supplier_id.integer' => 'El campo ID de proveedor debe ser un número entero.',
                    'supplier_name.required' => 'El campo Nombre de proveedor es obligatorio.',
                    'supplier_name.string' => 'El campo Nombre de proveedor debe ser una cadena de texto.',
                    'supplier_name.max' => 'El campo Nombre de proveedor no debe superar los 55 caracteres.',
                    'currency.string' => 'El campo Moneda debe ser una cadena de texto.',
                    'currency.max' => 'El campo Moneda no debe superar los 3 caracteres.',
                    'exchange.numeric' => 'El campo Tipo de cambio debe ser un valor numérico.',
                    'affected.required' => 'El campo Afectado es obligatorio.',
                    'affected.numeric' => 'El campo Afectado debe ser un valor numérico.',
                    'non_affected.required' => 'El campo No afectado es obligatorio.',
                    'non_affected.numeric' => 'El campo No afectado debe ser un valor numérico.',
                    'exonerated.required' => 'El campo Exonerado es obligatorio.',
                    'exonerated.numeric' => 'El campo Exonerado debe ser un valor numérico.',
                    'total.required' => 'El campo Total es obligatorio.',
                    'total.numeric' => 'El campo Total debe ser un valor numérico.',
                    'product_discount.numeric' => 'El campo Descuento de producto debe ser un valor numérico.',
                    'order_discount_id.max' => 'El campo ID de descuento de pedido no debe superar los 20 caracteres.',
                    'order_discount.numeric' => 'El campo Descuento de pedido debe ser un valor numérico.',
                    'total_discount.numeric' => 'El campo Descuento total debe ser un valor numérico.',
                    'product_tax.numeric' => 'El campo Impuesto de producto debe ser un valor numérico.',
                    'order_tax_id.max' => 'El campo ID de impuesto de pedido no debe superar los 20 caracteres.',
                    'order_tax.numeric' => 'El campo Impuesto de pedido debe ser un valor numérico.',
                    'total_tax.numeric' => 'El campo Impuesto total debe ser un valor numérico.',
                    'tax_ICBPER.required' => 'El campo Impuesto ICBPER es obligatorio.',
                    'tax_ICBPER.numeric' => 'El campo Impuesto ICBPER debe ser un valor numérico.',
                    'mult_ICBPER.required' => 'El campo Multiplicador ICBPER es obligatorio.',
                    'mult_ICBPER.numeric' => 'El campo Multiplicador ICBPER debe ser un valor numérico.',
                    'subtotal.required' => 'El campo Subtotal es obligatorio.',
                    'subtotal.numeric' => 'El campo Subtotal debe ser un valor numérico.',
                    'grand_total.required' => 'El campo Gran total es obligatorio.',
                    'grand_total.numeric' => 'El campo Gran total debe ser un valor numérico.',
                    'total_items.integer' => 'El campo Total de ítems debe ser un número entero.',
                    'total_quantity.numeric' => 'El campo Cantidad total debe ser un valor numérico.',
                    'note.string' => 'El campo Nota debe ser una cadena de texto.',
                    'note.max' => 'El campo Nota no debe superar los 1000 caracteres.',
                    'rounding.numeric' => 'El campo Redondeo debe ser un valor numérico.',
                    'attachment.string' => 'El campo Adjunto debe ser una cadena de texto.',
                    'attachment.max' => 'El campo Adjunto no debe superar los 255 caracteres.',
                    'expiration_date.date' => 'El campo Fecha de vencimiento debe ser una fecha válida.',
                    'canal_id.integer' => 'El campo ID de canal debe ser un número entero.',
                    'payment_method_id.integer' => 'El campo ID de método de pago debe ser un número entero.',
                    'estado.required' => 'El campo Estado es obligatorio.',
                    'estado.integer' => 'El campo Estado debe ser un número entero.',
                    'status.integer' => 'El campo Estado debe ser un número entero.',

                    //Mensajes de validacion para items:
                    'items.*.purchase_order_header_id.required' => 'El campo ID de compra es obligatorio para todos los elementos.',
                    'items.*.product_id.integer' => 'El campo ID de producto debe ser un número entero para todos los elementos.',
                    'items.*.code.string' => 'El campo Código debe ser una cadena de texto para todos los elementos.',
                    'items.*.code.max' => 'El campo Código no debe superar los 50 caracteres para todos los elementos.',
                    'items.*.name.string' => 'El campo Nombre debe ser una cadena de texto para todos los elementos.',
                    'items.*.name.max' => 'El campo Nombre no debe superar los 255 caracteres para todos los elementos.',
                    'items.*.quantity.required' => 'El campo Cantidad es obligatorio para todos los elementos.',
                    'items.*.quantity.numeric' => 'El campo Cantidad debe ser un valor numérico para todos los elementos.',
                    'items.*.unit_price.required' => 'El campo Precio unitario es obligatorio para todos los elementos.',
                    'items.*.unit_price.numeric' => 'El campo Precio unitario debe ser un valor numérico para todos los elementos.',
                    'items.*.affect_price.required' => 'El campo Precio afectado es obligatorio para todos los elementos.',
                    'items.*.affect_price.numeric' => 'El campo Precio afectado debe ser un valor numérico para todos los elementos.',
                    'items.*.non_affected_price.required' => 'El campo Precio no afectado es obligatorio para todos los elementos.',
                    'items.*.non_affected_price.numeric' => 'El campo Precio no afectado debe ser un valor numérico para todos los elementos.',
                    'items.*.exonerated_price.required' => 'El campo Precio exonerado es obligatorio para todos los elementos.',
                    'items.*.exonerated_price.numeric' => 'El campo Precio exonerado debe ser un valor numérico para todos los elementos.',
                    'items.*.gratuito_price.numeric' => 'El campo Precio gratuito debe ser un valor numérico para todos los elementos.',
                    'items.*.price_affected_item.required' => 'El campo Precio afectado por ítem es obligatorio para todos los elementos.',
                    'items.*.price_affected_item.numeric' => 'El campo Precio afectado por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.price_no_affected_item.required' => 'El campo Precio no afectado por ítem es obligatorio para todos los elementos.',
                    'items.*.price_no_affected_item.numeric' => 'El campo Precio no afectado por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.price_exonerated_item.required' => 'El campo Precio exonerado por ítem es obligatorio para todos los elementos.',
                    'items.*.price_exonerated_item.numeric' => 'El campo Precio exonerado por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.price_gratuito_item.numeric' => 'El campo Precio gratuito por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.discount.string' => 'El campo Descuento debe ser una cadena de texto para todos los elementos.',
                    'items.*.discount.max' => 'El campo Descuento no debe superar los 20 caracteres para todos los elementos.',
                    'items.*.item_discount.numeric' => 'El campo Descuento por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.prr_discount.required' => 'El campo Descuento PRR es obligatorio para todos los elementos.',
                    'items.*.prr_discount.numeric' => 'El campo Descuento PRR debe ser un valor numérico para todos los elementos.',
                    'items.*.tax.required' => 'El campo Impuesto es obligatorio para todos los elementos.',
                    'items.*.tax.numeric' => 'El campo Impuesto debe ser un valor numérico para todos los elementos.',
                    'items.*.item_tax.numeric' => 'El campo Impuesto por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.tax_gratuito.numeric' => 'El campo Impuesto gratuito debe ser un valor numérico para todos los elementos.',
                    'items.*.item_tax_gratuito.numeric' => 'El campo Impuesto por ítem gratuito debe ser un valor numérico para todos los elementos.',
                    'items.*.igv.required' => 'El campo IGV es obligatorio para todos los elementos.',
                    'items.*.igv.max' => 'El campo IGV no debe superar los 20 caracteres para todos los elementos.',
                    'items.*.tax_ICBPER.required' => 'El campo Impuesto ICBPER es obligatorio para todos los elementos.',
                    'items.*.tax_ICBPER.numeric' => 'El campo Impuesto ICBPER debe ser un valor numérico para todos los elementos.',
                    'items.*.quantity_ICBPER.required' => 'El campo Cantidad ICBPER es obligatorio para todos los elementos.',
                    'items.*.quantity_ICBPER.integer' => 'El campo Cantidad ICBPER debe ser un número entero para todos los elementos.',
                    'items.*.subtotal.required' => 'El campo Subtotal es obligatorio para todos los elementos.',
                    'items.*.subtotal.numeric' => 'El campo Subtotal debe ser un valor numérico para todos los elementos.',
                    'items.*.real_unit_price.numeric' => 'El campo Precio unitario real debe ser un valor numérico para todos los elementos.',
                    'items.*.cost.numeric' => 'El campo Costo debe ser un valor numérico para todos los elementos.',
                    'items.*.tax_method.required' => 'El campo Método de impuesto es obligatorio para todos los elementos.',
                    'items.*.tax_method.integer' => 'El campo Método de impuesto debe ser un número entero para todos los elementos.',
                    'items.*.status.numeric' => 'El campo Estado debe ser un valor numérico para todos los elementos.',
                ]
            );

            $store = new PurchaseOrderCreateDto();

            $store->ruc = $business_ruc;

            $store->date = $request["date"];
            $store->invoice_id = $request["invoice_id"];
            $store->supplier_id = $request["supplier_id"];
            $store->supplier_name = $request["supplier_name"];
            $store->currency = $request["currency"];
            $store->exchange = $request["exchange"];
            $store->affected = $request["affected"];
            $store->non_affected = $request["non_affected"];
            $store->exonerated = $request["exonerated"];
            $store->total = $request["total"];
            $store->product_discount = $request["product_discount"];
            $store->order_discount_id = $request["order_discount_id"];
            $store->order_discount = $request["order_discount"];
            $store->total_discount = $request["total_discount"];
            $store->product_tax = $request["product_tax"];
            $store->order_tax_id = $request["order_tax_id"];
            $store->order_tax = $request["order_tax"];
            $store->total_tax = $request["total_tax"];
            $store->tax_ICBPER = $request["tax_ICBPER"];
            $store->mult_ICBPER = $request["mult_ICBPER"];
            $store->subtotal = $request["subtotal"];
            $store->grand_total = $request["grand_total"];
            $store->total_items = count($request['items']);
            $store->total_quantity = $request["total_quantity"];
            $store->note = $request["note"];
            $store->rounding = $request["rounding"];
            $store->attachment = $request["attachment"];
            $store->expiration_date = $request["expiration_date"];
            $store->canal_id = $request["canal_id"];
            $store->payment_method_id = $request["payment_method_id"];
            $store->estado = $request["estado"];

            //Campos de Auditoria
            $store->created_by = $request['created_by'];
            $store->created_at = date('Y-m-d h:m:s');
            $store->status = 1;



            //items

            $store->items = (array)$request['items']; // Esto funciona?

            $result = $this->purchaseOrderRepository->store($store);

            return $this->successResponse($result, "La orden de compra se agrego correctamente.", 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 422);
        }
    }

    public function update(int $id, Request $request)
    {
        try {
            if ($this->errorConnection != '') {
                return $this->messageError($this->errorConnection, 500);
            }

            $business_information = $request['business_information'];
            $business_ruc = $business_information['ruc_number'];

            $this->validate(
                $request,
                [
                    'date' => 'required|date',
                    'invoice_id' => 'nullable|string|max:12',
                    'supplier_id' => 'required|integer',
                    'supplier_name' => 'required|string|max:55',
                    'currency' => 'nullable|string|max:3',
                    'exchange' => 'nullable|numeric',
                    'affected' => 'required|numeric',
                    'non_affected' => 'required|numeric',
                    'exonerated' => 'required|numeric',
                    'total' => 'required|numeric',
                    'product_discount' => 'nullable|numeric',
                    'order_discount_id' => 'nullable|string|max:20',
                    'order_discount' => 'nullable|numeric',
                    'total_discount' => 'nullable|numeric',
                    'product_tax' => 'nullable|numeric',
                    'order_tax_id' => 'nullable|string|max:20',
                    'order_tax' => 'nullable|numeric',
                    'total_tax' => 'nullable|numeric',
                    'tax_ICBPER' => 'required|numeric',
                    'mult_ICBPER' => 'required|numeric',
                    'subtotal' => 'required|numeric',
                    'grand_total' => 'required|numeric',
                    // 'total_items' => 'nullable|integer',
                    'total_quantity' => 'nullable|numeric',
                    'note' => 'nullable|string|max:1000',
                    'rounding' => 'nullable|numeric',
                    'attachment' => 'nullable|string|max:255',
                    'expiration_date' => 'nullable|date',
                    'canal_id' => 'nullable|integer',
                    'payment_method_id' => 'nullable|integer',
                    'estado' => 'required|integer',
                    'status' => 'nullable|integer',

                    'items.*.purchase_order_header_id' => 'required|integer|max:15',
                    'items.*.product_id' => 'integer|nullable', //Este campo debe validarse que exista
                    'items.*.code' => 'string|nullable|max:50',
                    'items.*.name' => 'string|nullable|max:255',
                    'items.*.quantity' => 'required|numeric',
                    'items.*.unit_price' => 'required|numeric',
                    'items.*.affect_price' => 'required|numeric',
                    'items.*.non_affected_price' => 'required|numeric',
                    'items.*.exonerated_price' => 'required|numeric',
                    'items.*.gratuito_price' => 'numeric|nullable',
                    'items.*.price_affected_item' => 'required|numeric',
                    'items.*.price_no_affected_item' => 'required|numeric',
                    'items.*.price_exonerated_item' => 'required|numeric',
                    'items.*.price_gratuito_item' => 'numeric|nullable',
                    'items.*.discount' => 'string|nullable|max:20',
                    'items.*.item_discount' => 'numeric|nullable',
                    'items.*.prr_discount' => 'required|numeric',
                    'items.*.tax' => 'required|numeric',
                    'items.*.item_tax' => 'numeric|nullable',
                    'items.*.tax_gratuito' => 'numeric|nullable',
                    'items.*.item_tax_gratuito' => 'numeric|nullable',
                    'items.*.igv' => 'required|string|max:20',
                    'items.*.tax_ICBPER' => 'required|numeric',
                    'items.*.quantity_ICBPER' => 'required|integer',
                    'items.*.subtotal' => 'required|numeric',
                    'items.*.real_unit_price' => 'numeric|nullable',
                    'items.*.cost' => 'numeric|nullable',
                    'items.*.tax_method' => 'required|integer',
                    'items.*.status' => 'nullable|integer',
                ],
                [   //Mensajes para las validaciones:
                    'date.required' => 'El campo Fecha es obligatorio.',
                    'date.date' => 'El campo Fecha debe ser una fecha válida.',
                    'invoice_id.max' => 'El campo ID de factura no debe superar los 12 caracteres.',
                    'supplier_id.required' => 'El campo ID de proveedor es obligatorio.',
                    'supplier_id.integer' => 'El campo ID de proveedor debe ser un número entero.',
                    'supplier_name.required' => 'El campo Nombre de proveedor es obligatorio.',
                    'supplier_name.string' => 'El campo Nombre de proveedor debe ser una cadena de texto.',
                    'supplier_name.max' => 'El campo Nombre de proveedor no debe superar los 55 caracteres.',
                    'currency.string' => 'El campo Moneda debe ser una cadena de texto.',
                    'currency.max' => 'El campo Moneda no debe superar los 3 caracteres.',
                    'exchange.numeric' => 'El campo Tipo de cambio debe ser un valor numérico.',
                    'affected.required' => 'El campo Afectado es obligatorio.',
                    'affected.numeric' => 'El campo Afectado debe ser un valor numérico.',
                    'non_affected.required' => 'El campo No afectado es obligatorio.',
                    'non_affected.numeric' => 'El campo No afectado debe ser un valor numérico.',
                    'exonerated.required' => 'El campo Exonerado es obligatorio.',
                    'exonerated.numeric' => 'El campo Exonerado debe ser un valor numérico.',
                    'total.required' => 'El campo Total es obligatorio.',
                    'total.numeric' => 'El campo Total debe ser un valor numérico.',
                    'product_discount.numeric' => 'El campo Descuento de producto debe ser un valor numérico.',
                    'order_discount_id.max' => 'El campo ID de descuento de pedido no debe superar los 20 caracteres.',
                    'order_discount.numeric' => 'El campo Descuento de pedido debe ser un valor numérico.',
                    'total_discount.numeric' => 'El campo Descuento total debe ser un valor numérico.',
                    'product_tax.numeric' => 'El campo Impuesto de producto debe ser un valor numérico.',
                    'order_tax_id.max' => 'El campo ID de impuesto de pedido no debe superar los 20 caracteres.',
                    'order_tax.numeric' => 'El campo Impuesto de pedido debe ser un valor numérico.',
                    'total_tax.numeric' => 'El campo Impuesto total debe ser un valor numérico.',
                    'tax_ICBPER.required' => 'El campo Impuesto ICBPER es obligatorio.',
                    'tax_ICBPER.numeric' => 'El campo Impuesto ICBPER debe ser un valor numérico.',
                    'mult_ICBPER.required' => 'El campo Multiplicador ICBPER es obligatorio.',
                    'mult_ICBPER.numeric' => 'El campo Multiplicador ICBPER debe ser un valor numérico.',
                    'subtotal.required' => 'El campo Subtotal es obligatorio.',
                    'subtotal.numeric' => 'El campo Subtotal debe ser un valor numérico.',
                    'grand_total.required' => 'El campo Gran total es obligatorio.',
                    'grand_total.numeric' => 'El campo Gran total debe ser un valor numérico.',
                    // 'total_items.integer' => 'El campo Total de ítems debe ser un número entero.',
                    'total_quantity.numeric' => 'El campo Cantidad total debe ser un valor numérico.',
                    'note.string' => 'El campo Nota debe ser una cadena de texto.',
                    'note.max' => 'El campo Nota no debe superar los 1000 caracteres.',
                    'rounding.numeric' => 'El campo Redondeo debe ser un valor numérico.',
                    'attachment.string' => 'El campo Adjunto debe ser una cadena de texto.',
                    'attachment.max' => 'El campo Adjunto no debe superar los 255 caracteres.',
                    'expiration_date.date' => 'El campo Fecha de vencimiento debe ser una fecha válida.',
                    'canal_id.integer' => 'El campo ID de canal debe ser un número entero.',
                    'payment_method_id.integer' => 'El campo ID de método de pago debe ser un número entero.',
                    'estado.required' => 'El campo Estado es obligatorio.',
                    'estado.integer' => 'El campo Estado debe ser un número entero.',
                    'status.integer' => 'El campo Estado debe ser un número entero.',

                    //Mensajes de validacion para items:
                    'items.*.purchase_order_header_id.required' => 'El campo ID de compra es obligatorio para todos los elementos.',
                    'items.*.product_id.integer' => 'El campo ID de producto debe ser un número entero para todos los elementos.',
                    'items.*.code.string' => 'El campo Código debe ser una cadena de texto para todos los elementos.',
                    'items.*.code.max' => 'El campo Código no debe superar los 50 caracteres para todos los elementos.',
                    'items.*.name.string' => 'El campo Nombre debe ser una cadena de texto para todos los elementos.',
                    'items.*.name.max' => 'El campo Nombre no debe superar los 255 caracteres para todos los elementos.',
                    'items.*.quantity.required' => 'El campo Cantidad es obligatorio para todos los elementos.',
                    'items.*.quantity.numeric' => 'El campo Cantidad debe ser un valor numérico para todos los elementos.',
                    'items.*.unit_price.required' => 'El campo Precio unitario es obligatorio para todos los elementos.',
                    'items.*.unit_price.numeric' => 'El campo Precio unitario debe ser un valor numérico para todos los elementos.',
                    'items.*.affect_price.required' => 'El campo Precio afectado es obligatorio para todos los elementos.',
                    'items.*.affect_price.numeric' => 'El campo Precio afectado debe ser un valor numérico para todos los elementos.',
                    'items.*.non_affected_price.required' => 'El campo Precio no afectado es obligatorio para todos los elementos.',
                    'items.*.non_affected_price.numeric' => 'El campo Precio no afectado debe ser un valor numérico para todos los elementos.',
                    'items.*.exonerated_price.required' => 'El campo Precio exonerado es obligatorio para todos los elementos.',
                    'items.*.exonerated_price.numeric' => 'El campo Precio exonerado debe ser un valor numérico para todos los elementos.',
                    'items.*.gratuito_price.numeric' => 'El campo Precio gratuito debe ser un valor numérico para todos los elementos.',
                    'items.*.price_affected_item.required' => 'El campo Precio afectado por ítem es obligatorio para todos los elementos.',
                    'items.*.price_affected_item.numeric' => 'El campo Precio afectado por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.price_no_affected_item.required' => 'El campo Precio no afectado por ítem es obligatorio para todos los elementos.',
                    'items.*.price_no_affected_item.numeric' => 'El campo Precio no afectado por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.price_exonerated_item.required' => 'El campo Precio exonerado por ítem es obligatorio para todos los elementos.',
                    'items.*.price_exonerated_item.numeric' => 'El campo Precio exonerado por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.price_gratuito_item.numeric' => 'El campo Precio gratuito por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.discount.string' => 'El campo Descuento debe ser una cadena de texto para todos los elementos.',
                    'items.*.discount.max' => 'El campo Descuento no debe superar los 20 caracteres para todos los elementos.',
                    'items.*.item_discount.numeric' => 'El campo Descuento por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.prr_discount.required' => 'El campo Descuento PRR es obligatorio para todos los elementos.',
                    'items.*.prr_discount.numeric' => 'El campo Descuento PRR debe ser un valor numérico para todos los elementos.',
                    'items.*.tax.required' => 'El campo Impuesto es obligatorio para todos los elementos.',
                    'items.*.tax.numeric' => 'El campo Impuesto debe ser un valor numérico para todos los elementos.',
                    'items.*.item_tax.numeric' => 'El campo Impuesto por ítem debe ser un valor numérico para todos los elementos.',
                    'items.*.tax_gratuito.numeric' => 'El campo Impuesto gratuito debe ser un valor numérico para todos los elementos.',
                    'items.*.item_tax_gratuito.numeric' => 'El campo Impuesto por ítem gratuito debe ser un valor numérico para todos los elementos.',
                    'items.*.igv.required' => 'El campo IGV es obligatorio para todos los elementos.',
                    'items.*.igv.max' => 'El campo IGV no debe superar los 20 caracteres para todos los elementos.',
                    'items.*.tax_ICBPER.required' => 'El campo Impuesto ICBPER es obligatorio para todos los elementos.',
                    'items.*.tax_ICBPER.numeric' => 'El campo Impuesto ICBPER debe ser un valor numérico para todos los elementos.',
                    'items.*.quantity_ICBPER.required' => 'El campo Cantidad ICBPER es obligatorio para todos los elementos.',
                    'items.*.quantity_ICBPER.integer' => 'El campo Cantidad ICBPER debe ser un número entero para todos los elementos.',
                    'items.*.subtotal.required' => 'El campo Subtotal es obligatorio para todos los elementos.',
                    'items.*.subtotal.numeric' => 'El campo Subtotal debe ser un valor numérico para todos los elementos.',
                    'items.*.real_unit_price.numeric' => 'El campo Precio unitario real debe ser un valor numérico para todos los elementos.',
                    'items.*.cost.numeric' => 'El campo Costo debe ser un valor numérico para todos los elementos.',
                    'items.*.tax_method.required' => 'El campo Método de impuesto es obligatorio para todos los elementos.',
                    'items.*.tax_method.integer' => 'El campo Método de impuesto debe ser un número entero para todos los elementos.',
                    'items.*.status.numeric' => 'El campo Estado debe ser un valor numérico para todos los elementos.',
                ]
            );

            $entry = new PurchaseOrderUpdateDto();

            $entry->ruc = $business_ruc;
            $entry->id = $id;
            $entry->date = $request["date"];
            $entry->invoice_id = $request["invoice_id"];
            $entry->supplier_id = $request["supplier_id"];
            $entry->supplier_name = $request["supplier_name"];
            $entry->currency = $request["currency"];
            $entry->exchange = $request["exchange"];
            $entry->affected = $request["affected"];
            $entry->non_affected = $request["non_affected"];
            $entry->exonerated = $request["exonerated"];
            $entry->total = $request["total"];
            $entry->product_discount = $request["product_discount"];
            $entry->order_discount_id = $request["order_discount_id"];
            $entry->order_discount = $request["order_discount"];
            $entry->total_discount = $request["total_discount"];
            $entry->product_tax = $request["product_tax"];
            $entry->order_tax_id = $request["order_tax_id"];
            $entry->order_tax = $request["order_tax"];
            $entry->total_tax = $request["total_tax"];
            $entry->tax_ICBPER = $request["tax_ICBPER"];
            $entry->mult_ICBPER = $request["mult_ICBPER"];
            $entry->subtotal = $request["subtotal"];
            $entry->grand_total = $request["grand_total"];
            $entry->total_items = count($request['items']);
            $entry->total_quantity = $request["total_quantity"];
            $entry->note = $request["note"];
            $entry->rounding = $request["rounding"];
            $entry->attachment = $request["attachment"];
            $entry->expiration_date = $request["expiration_date"];
            $entry->canal_id = $request["canal_id"];
            $entry->payment_method_id = $request["payment_method_id"];
            $entry->estado = $request["estado"];

            // Campos de Auditoria
            $entry->created_by = $request['created_by'];
            $entry->created_at = date('Y-m-d h:m:s');
            $entry->status = 1;

            // items
            $entry->items = (array)$request['items']; // ¿Esto funciona?

            $this->purchaseOrderRepository->update($entry);

            $result = $this->purchaseOrderRepository->find($id, $business_ruc);

            return $this->successResponse($result, "La orden de compra se actualizo correctamente.", 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 422);
        }
    }

    public function destroy(int $id, Request $request)
    {

        if ($this->errorConnection != '') {
            return $this->messageError($this->errorConnection, 500);
        }
        $business_information = $request['business_information'];
        $business_ruc = $business_information['ruc_number'];

        $result = $this->purchaseOrderRepository->find($id, $business_ruc);

        if ($result) {
            $this->purchaseOrderRepository->destroy($id);

            return $this->successResponse($id, "Orden de compra eliminado correctamente.", 200);
        } else {
            return $this->errorResponse("Orden de compra inexistente.", 404);
        }  

    }
}
