<?php
namespace App\Http\Controllers;

use App\DTOs\PriceProduct\PriceProductCreateDto;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DTOs\Products\ProductCreateDto;
use App\DTOs\Products\ProductUpdateDto;
use App\Repositories\Interfaces\IProductRepository;
use Exception;
use App\Models\Product;
use App\Models\ComboItem;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Symfony\Component\Console\Output\ConsoleOutput;

class ProductController extends Controller
{

    private $productRepository;

    private $errorConnection = "";

    public function __construct(Request $request, IProductRepository $productRepository)
    {
        if (isset($request['business_information'])) {
            $business_information = $request['business_information'];
            $ruc = $business_information['ruc_number'];

            $existe_conexion = $this->setDBClient($ruc);

            if($existe_conexion['error']){
                $this->errorConnection=$existe_conexion['message'];
                return $this->messageError("Se espera un número de RUC válido.", 201);
            } else {
                $this->productRepository = $productRepository;
            }
        } else {
            $this->errorConnection='Error en la conexión. Se espera número de RUC';
        }
    }

    public function index(Request $request)
    {
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201);
        }

        $result = $this->productRepository->paginate($request->all());

        if ($result) {
            return $this->successResponse($result, "Productos encontrados correctamente", 201);
        } else {
            return $this->errorResponse("No hay Productos", 201);
        }

    }

    public function show(int $id)
    {
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201);
        }

        $result = $this->productRepository->find($id);

        if ($result) {
            $items = ComboItem::where('product_id', '=', $id)->get();
            $result['items'] = $items;

            return $this->successResponse($result, "Producto encontrado correctamente", 201);
        } else {
            return $this->errorResponse("Producto no encontrado", 201);
        }
    }

    public function store(Request $request)
    {
        try {
            if($this->errorConnection != ''){
                return $this->messageError($this->errorConnection, 201);
            }

            $code = $request['code'];
            $codeExist = Product::where('code', '=', $code)->get();

            if (count($codeExist) == 0) {
                $this->validate($request, [
                    'type' => 'required|max:20',
                    'name' => 'required|max:255',
                    'code' => 'required|max:50',
                    'short_name' => 'required|max:50',
                    'requires_lot_number' => 'required|max:1',
                    'requires_serial_number' => 'required|max:1',
                    'currency' => 'max:3',
                    'price' => 'required',
                    'image' => 'max:255',
                    'tax' => 'max:20',
                    'tax_method' => 'max:1',
                    'barcode_symbology' => 'required|max:20',
                    'custom_field_1' => 'max:250',
                    'custom_field_2' => 'max:250',
                    'custom_field_3' => 'max:250',
                    'presentation' => 'max:250'
                ], [
                    'type.required' => 'El tipo del Producto es requerido',
                    'code.required' => 'El código de Producto es requerido',
                    'code.max' => 'El código de Producto no puede superar los :max caracteres',
                    'name.required' => 'El nombre del Producto es requerido',
                    'short_name.required' => 'La nombre corto del Producto es requerido',
                    'requires_lot_number.required' => 'El requiere Numero de Lote del Producto es requerido',
                    'requires_serial_number.required' => 'El requiere Numero de Serie del Producto es requerido',
                    'price.required' => 'El precio del Producto es requerido',
                    'barcode_symbology.required' => 'La simbologia CodBar del Producto es requerido'
                ]);
                $data = $request->except(array('business_information','items'));
                $data['estado'] = 1;
                $data['category_id'] = $data['category_id']!=""?$data['category_id']:null;
                $store = new ProductCreateDto($data);

                $result = $this->productRepository->store($store);

                if (isset($request['items'])) {
                    $items = $request['items'];
                    $total_items = count($items);

                    if ($total_items != 0) {
                        $product_id = $result['id'];

                        foreach ($items as $item) {
                            $item_code = $item['item_code'];
                            $quantity = $item['quantity'];

                            $product_data = Product::where('code', '=', $item_code)->get();

                            $price = $product_data[0]->price;
                            $cost = $product_data[0]->cost;

                            $data = array(
                                'product_id'=>$product_id,
                                'item_code'=>$item_code,
                                'quantity'=>$quantity,
                                'price'=>$price,
                                'cost'=>$cost
                            );

                            ComboItem::insert($data);
                        }
                    }
                }

                return $this->successResponse($result, "Producto creado correctamente", 201);
            } else {
                return $this->errorResponse("Codigo de Producto ingresado ya existe", 201);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 201);
        }
    }

    public function update(int $id, Request $request)
    {
        try {
            if($this->errorConnection != ''){
                return $this->messageError($this->errorConnection, 201);
            }

            $result = $this->productRepository->find($id);

            if ($result) {
                $code = $request['code'];
                $codeExist = Product::where('id', '!=', $id)->where('code', '=', $code)->get();

                if (count($codeExist) == 0) {
                    $this->validate($request, [
                        'code' => [
                            'required',
                            'max:50'
                        ],
                        'name' => 'required|max:255',
                        'short_name' => 'required|max:50',
                        'requires_lot_number' => 'required|max:1',
                        'requires_serial_number' => 'required|max:1',
                        'currency' => 'max:3',
                        'price' => 'required',
                        'image' => 'max:255',
                        'tax' => 'max:20',
                        'tax_method' => 'max:1',
                        'barcode_symbology' => 'required|max:20',
                        'type' => 'required|max:20',
                        'custom_field_1' => 'max:250',
                        'custom_field_2' => 'max:250',
                        'custom_field_3' => 'max:250',
                        'presentation' => 'max:250'
                    ], [
                        'code.required' => 'El código de Producto es requerido',
                        'code.max' => 'El código de Producto no puede superar los :max caracteres',
                        'name.required' => 'El nombre del Producto es requerido',
                        'short_name.required' => 'La nombre corto del Producto es requerido',
                        'requires_lot_number.required' => 'El requiere Numero de Lote del Producto es requerido',
                        'requires_serial_number.required' => 'El requiere Numero de Serie del Producto es requerido',
                        'price.required' => 'El precio del Producto es requerido',
                        'barcode_symbology.required' => 'La simbologia CodBar del Producto es requerido',
                        'type.required' => 'El tipo del Producto es requerido'
                    ]);

                    $data = $request->except(array('business_information','items'));

                    $data['id'] = $id;
                    $data['estado'] = 1;

                    $entry = new ProductUpdateDto($data);

                    $this->productRepository->update($entry);

                    if (isset($request['items'])) {
                        $items = $request['items'];
                        $total_items = count($items);

                        if ($total_items != 0) {
                            ComboItem::where('product_id', '=', $id)->delete();

                            $product_id = $id;

                            foreach ($items as $item) {
                                $item_code = $item['item_code'];
                                $quantity = $item['quantity'];

                                $product_data = Product::where('code', '=', $item_code)->get();

                                $price = $product_data[0]->price;
                                $cost = $product_data[0]->cost;

                                $data = array(
                                    'product_id'=>$product_id,
                                    'item_code'=>$item_code,
                                    'quantity'=>$quantity,
                                    'price'=>$price,
                                    'cost'=>$cost
                                );

                                ComboItem::insert($data);
                            }
                        }
                    }

                    $result = $this->productRepository->find($id);

                    return $this->successResponse($result, "Producto actualizado correctamente", 201);
                } else {
                    return $this->errorResponse("Codigo de Producto ingresado ya existe", 201);
                }
            } else {
                return $this->errorResponse("Producto ingresado no existe", 201);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 201);
        }
    }

    public function destroy(int $id)
    {
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201);
        }

        $result = $this->productRepository->find($id);

        if ($result) {
            ComboItem::where('product_id', '=', $id)->delete();

            $this->productRepository->destroy($id);

            return $this->successResponse($id, "Producto eliminado correctamente", 201);
        } else {
            return $this->errorResponse("Producto ingresado no existe", 201);
        }
    }
    // public function validarIdProducto(int $category_id)
    // {
    //     if($this->errorConnection != '') {
    //         return $this->messageError($this->errorConnection, 201)
    //     }
    //     $result = $this->productRepository->find($category_id);

    //     if($result) {
    //         ComboItem::where('category_id', '=', $id)->delete();
    //         $this->productRepository->destroy($category_id);
    //         return $this->successResponse($category_id, "No ha seleccionado categoria", 201)

    //     }else{
    //         return $this->errorResponse("Categoria Ingresado exiosamente");
    //     }
    // }
    // public function ajax()
    // {
    //     $productRepository = productRepository::all();
    //     $
    // }
    public function priceProductListByIdProduct(Request $request){
        try{
            if($this->errorConnection != ''){
                return $this->messageError($this->errorConnection, 201);
            }
            $this->validate($request,[
                "product_id"=>'required|integer'
            ],
            [
                'product_id.required' => "El identificador del producto es requerido.",
                'product_id.integer' => "El identificador del producto debe ser un número entero.",

            ]);
            $result = $this->productRepository->find($request['product_id']);
            if(!$result){
                return $this->errorResponse(['product_id'=>'El producto no existe.'], 402);
            }
            $ruc_business = $request['business_information']['ruc_number'];
            $data = $this->productRepository->priceProductByCustomersTypeList($request->all(),$ruc_business);
            if(count($data) == 0){
                return $result;
            }
            return $this->successResponse($data,'Lista de precio encontrada.',200);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 422);
        }
    }
    public function priceProductStoreByIdProduct(Request $request){
        try{
            if($this->errorConnection != ''){
                return $this->messageError($this->errorConnection, 201);
            }
            $this->validate($request, [
                "product_id" => 'required|integer',
                "created_by" => 'required|integer',
                "price" => 'required|array', // Asegurarse de que "price" sea un arreglo
                "price.*.customer_type_id" => 'required|integer',
                "price.*.price" => 'required|regex:/^\d+(\.\d{1,2})?$/',
            ], [
                'product_id.required' => "El identificador del producto es requerido.",
                'product_id.integer' => "El identificador del producto debe ser un número entero.",
                'created_by.required' => "El el ususario de creación es requerido.",
                'created_by.integer' => "El el ususario de creación debe ser un número entero.",
                'price.required' => "La matriz de precios es requerida.",
                'price.array' => "La matriz de precios debe ser un arreglo.",
                'price.*.customer_type_id.required' => "El campo 'customer_type_id' es requerido.",
                'price.*.customer_type_id.integer' => "El campo 'customer_type_id' debe ser un número entero.",
                'price.*.price.required' => "El campo 'price' es requerido.",
                'price.*.price.regex' => "El campo 'price' debe ser un precio válido.",
            ]);

            $result = $this->productRepository->find($request['product_id']);
            if(!$result){
                return $this->errorResponse(['product_id'=>'El producto no existe.'], 402);
            }

            $ruc_business = $request['business_information']['ruc_number'];

            // Crear un arreglo para almacenar los datos transformados
            $dataErrores=[];
            foreach ($request['price'] as $priceData) {
                $client = new Client();
                    // URL del servicio externo
                    $url = config('constantes.api_ventas')."customer_type/".$priceData['customer_type_id'];
                    // Realizar una solicitud POST con los datos como JSON
                    $response = $client->get($url, [
                        'json' => $request->all(),
                    ]);
                    // Obtener el cuerpo de la respuesta
                    $responseBody = $response->getBody();
                    // Puedes decodificar la respuesta JSON si es necesario
                    $data = json_decode($responseBody, true);
                    if($data['result']=='failed'){
                        $dataErrores[] = $data['message'].": ".$priceData['customer_type_id'];
                    }
            }
            if(count($dataErrores)>0){
                return $this->errorResponse(['customer_type_id' => $dataErrores], 422);
            }
            $data = [];

            foreach ($request['price'] as $priceData) {
                    $PriceProductCreateDto = new PriceProductCreateDto([
                        'ruc'=>$ruc_business,
                        'product_id' => $request['product_id'],
                        'customer_type_id' => $priceData['customer_type_id'],
                        'price' => (float) $priceData['price'], // Convertir el precio a float
                        'created_by' => $request['created_by'],
                        'created_at' => Carbon::now(),
                        'status' => 1
                    ]);
                    $data[] = $this->productRepository->priceProductByCustomersTypeStore($PriceProductCreateDto);
            }
            return $data;
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 422);
        }
    }
    public function priceProductFindByProductAndCustomerType(int $product_id, $customer_type_id = NULL, Request $request){
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201);
        }
        $ruc_business = $request['business_information']['ruc_number'];
        $result = $this->productRepository->find($request['product_id']);
        if(!$result){
            return $this->errorResponse(['product_id'=>'El producto no existe.'], 402);
        }
        if($customer_type_id == null && $customer_type_id == 0){
            $result =  Product::select('id as producto_id','name as producto_name','price')->where('id',$product_id)->first();
            return $this->successResponse($result,'Precio del producto por el tipo de cliente encontrado.',200);
        }
        $client = new Client();
        // URL del servicio externo
        $url = config('constantes.api_ventas')."customer_type/".$customer_type_id;
        // Realizar una solicitud POST con los datos como JSON
        $response = $client->get($url, [
            'json' =>["business_information"=> $request['business_information']],
        ]);
        // Obtener el cuerpo de la respuesta
        $responseBody = $response->getBody();
        // Puedes decodificar la respuesta JSON si es necesario
        $data = json_decode($responseBody, true);
        if($data['result']=='failed'){
                return $this->errorResponse(['customer_type_id' => $data['message']], 422);
        }

        $data = $this->productRepository->priceProductByCustomersTypeFind($product_id,$customer_type_id, $ruc_business);
        if($data){
            return $this->successResponse($data,'Precio del producto por el tipo de cliente encontrado.',200);
        }else{
            return $this->errorResponse('No se encontró el precio del producto para el tipo de cliente indicado.',404);
        }
    }
}
